<?php
namespace App\Http\Controllers\Apis;

use App\Http\Controllers\Controller;
use App\Models\Channel;
use App\Models\NewsLanguage;
use App\Models\Post;
use App\Models\Topic;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;

class SuggestionPostController extends Controller
{
    /******* Get Suggestions *******/
    public function getsuggestion(Request $request)
    {
        $searchQuery = $request->input('search');
        $perPage     = $request->get('per_page', 5);

        $newsLanguageId   = $request->news_language_id;
        $newsLanguageCode = null;

        if ($newsLanguageId) {
            $newsLanguage = NewsLanguage::find($newsLanguageId);
            if ($newsLanguage) {
                $newsLanguageCode = $newsLanguage->code;
            }
        }
        // Split query into keywords
        $keywords = preg_split('/\s+/', $searchQuery, -1, PREG_SPLIT_NO_EMPTY);

        // Posts
        $posts = Post::select(
            'image',
            'video_thumb',
            'type',
            'title',
            'slug',
            DB::raw("'post' as source")
        )
            ->where('status', 'active')
            ->whereIn('type', ['video', 'youtube', 'post','audio'])
            ->when($newsLanguageId, function ($query) use ($newsLanguageId) {
                $query->where('news_language_id', $newsLanguageId);
            })
            ->when($keywords, function ($query) use ($keywords) {
                $query->where(function ($q) use ($keywords) {
                    foreach ($keywords as $word) {
                        $q->orWhere('title', 'LIKE', "%{$word}%");
                    }
                });
            });

        // Channels
        $channels = Channel::select(
            DB::raw('CONCAT("' . url('storage/images/') . '/", logo) as image'),
            'name as title',
            'slug',
            DB::raw('"" as video_thumb'),
            DB::raw('"" as type'),
            DB::raw("'channel' as source")
        )
            ->when($newsLanguageId, function ($query) use ($newsLanguageId) {
                $query->where('news_language_id', $newsLanguageId);
            })
            ->when($keywords, function ($query) use ($keywords) {
                $query->where(function ($q) use ($keywords) {
                    foreach ($keywords as $word) {
                        $q->orWhere('name', 'LIKE', "%{$word}%");
                    }
                });
            });

        // Topics
        $topics = Topic::select(
            DB::raw('"" as image'),
            DB::raw('"" as video_thumb'),
            DB::raw('"" as type'),
            'name as title',
            'slug',
            DB::raw("'topic' as source")
        )
            ->when($newsLanguageId, function ($query) use ($newsLanguageId) {
                $query->where('news_language_id', $newsLanguageId);
            })
            ->when($keywords, function ($query) use ($keywords) {
                $query->where(function ($q) use ($keywords) {
                    foreach ($keywords as $word) {
                        $q->orWhere('name', 'LIKE', "%{$word}%");
                    }
                });
            });

        // Merge all queries
        $suggestions = $posts
            ->union($channels)
            ->union($topics)
            ->orderBy('title', 'asc')
            ->paginate($perPage);

        $result = $suggestions->map(function ($suggestion) {
            return [
                'image'  => $suggestion->image ?: $suggestion->video_thumb,
                'title'  => $suggestion->title,
                'slug'   => $suggestion->slug,
                'source' => $suggestion->source,
            ];
        });

        return response()->json([
            'error'   => false,
            'message' => 'Get suggestion successfully.',
            'data'    => $result,
        ]);
    }

    /****** Get Search Data ******/
    public function search(Request $request)
    {
        $perPage     = $request->get('per_page', 10);
        $searchQuery = $request->input('search', '');
        $userId      = Auth::check() ? Auth::user()->id : 0;

        $newsLanguageId   = $request->news_language_id;
        $newsLanguageCode = null;
        if ($newsLanguageId) {
            $newsLanguage = NewsLanguage::find($newsLanguageId);
            if ($newsLanguage) {
                $newsLanguageCode = $newsLanguage->code;
            }
        }

        $keywords = preg_split('/\s+/', $searchQuery, -1, PREG_SPLIT_NO_EMPTY);

        $getPosts = Post::select(
            'posts.id', 'posts.favorite', 'posts.image', 'posts.type', 'posts.video_thumb',
            'posts.title', 'posts.slug', 'posts.view_count',
            'channels.name as channel_name', 'channels.logo as channel_logo', 'channels.slug as channel_slug',
            'posts.status', 'topics.name as topic_name', 'topics.slug as topic_slug',
            'posts.pubdate', 'posts.publish_date'
        )
            ->selectRaw('IF(favorites.user_id IS NOT NULL, 1, 0) as is_favorite')
            ->join('channels', 'posts.channel_id', '=', 'channels.id')
            ->join('topics', 'posts.topic_id', '=', 'topics.id')
            ->leftJoin('favorites', function ($join) use ($userId) {
                $join->on('posts.id', '=', 'favorites.post_id')
                    ->where('favorites.user_id', '=', $userId);
            })
            ->where('posts.status', 'active')
            ->when($newsLanguageId, function ($query) use ($newsLanguageId) {
                $query->where('posts.news_language_id', $newsLanguageId);
            })
            ->when($keywords, function ($query) use ($keywords) {
                $query->where(function ($q) use ($keywords) {
                    foreach ($keywords as $word) {
                        $q->orWhere('posts.slug', 'LIKE', "%{$word}%")
                            ->orWhere('posts.title', 'LIKE', "%{$word}%")
                            ->orWhere('channels.name', 'LIKE', "%{$word}%")
                            ->orWhere('topics.name', 'LIKE', "%{$word}%");
                    }
                });
            })
            ->orderBy('posts.publish_date', 'DESC')
            ->limit($perPage) // optional, to control max results
            ->get();          // <-- use get() instead of paginate()

        $getPosts->transform(function ($item) {
            $item->image        = $item->image ? url('storage/posts/' . $item->image) : '';
            $item->video_thumb  = $item->video_thumb ?? '';
            $item->publish_date = $item->publish_date ? \Carbon\Carbon::parse($item->publish_date)->diffForHumans() : '';
            $item->channel_logo = url('storage/images/' . $item->channel_logo);
            $item->search       = true;
            return $item;
        });

        return response()->json([
            'error'   => false,
            'message' => 'Fetch search result successfully.',
            'data'    => $getPosts, // plain array of posts
        ]);
    }
}
